#include "ViewFrustum.h"
#include <math.h>
#include "SCamera.h"
//******************************************************************
//
// The ViewFrustum Class
//
//-----------------------------------------------------------------------
//
// constructor
//
ViewFrustum::ViewFrustum()
{
	//initialize the planes pointers
	plane[0]=&m_nr;
	plane[1]=&m_lf;
	plane[2]=&m_rg;
	plane[3]=&m_up;
	plane[4]=&m_dn;
	plane[5]=&m_fr;
}

//-----------------------------------------------------------------------
//
// set the prespecive vals for successfull computing
// of the clipping planes
//
void ViewFrustum::SetPerspectiveSpecs(FLOAT fovy, FLOAT Aspect, 
									FLOAT zn, FLOAT zf) 
{
	m_fovy=fovy;
	m_asp=Aspect;
	m_zn=zn;
	m_zf=zf;

	m_cpos=D3DXVECTOR3(0,0,0); //camera position
	m_cxpos=D3DXVECTOR3(1,0,0);//camera x help position
	m_cypos=D3DXVECTOR3(0,1,0);//camera y help position
	
	m_nypos=D3DXVECTOR3(0,1.0f/(float)tan(m_fovy),m_zn); //near plane y pos
	m_nxpos=D3DXVECTOR3(m_asp/(float)tan(m_fovy),0,m_zn); //near plane x

	m_nxopos=-m_nxpos;
	m_nyopos=-m_nypos;
	m_nyopos.z=m_nxopos.z=m_zn;

	m_ncpos=D3DXVECTOR3(0,0,m_zn); //near plane center
	m_fcpos=D3DXVECTOR3(0,0,m_zf); //far near plane center
}

//-----------------------------------------------------------------------
//
// gets the curent view transofrmation matrix and
// computes the planes (according to the perspective specs)
// extracts the possision of the camera
// 
void ViewFrustum::SetViewTransform(D3DXMATRIX &mtx)
{
	D3DXMATRIX imtx;
	float dt;
	D3DXMatrixInverse(&imtx,&dt,&mtx);

	D3DXVECTOR3 cpos; //camera position
	D3DXVECTOR3 cxpos;//camera x help position
	D3DXVECTOR3 cypos;//camera y help position
	
	D3DXVECTOR3 nypos; //near plane y pos
	D3DXVECTOR3 nxpos; //near plane x

	D3DXVECTOR3 nyopos; //near plane y pos
	D3DXVECTOR3 nxopos; //near plane x


	D3DXVECTOR3 ncpos; //near plane center
	D3DXVECTOR3 fcpos; //far near plane center

	D3DXVECTOR3 cl(0,0,1);

	//transform everything from camera space to world space
	D3DXVec3TransformCoord(&m_current,&cl,&imtx);
	D3DXVec3Normalize(&m_current,&m_current);
	D3DXVec3TransformCoord(&cpos,&m_cpos,&imtx);
	D3DXVec3TransformCoord(&cxpos,&m_cxpos,&imtx);
	D3DXVec3TransformCoord(&cypos,&m_cypos,&imtx);
	D3DXVec3TransformCoord(&nxpos,&m_nxpos,&imtx);
	D3DXVec3TransformCoord(&nypos,&m_nypos,&imtx);

	D3DXVec3TransformCoord(&nxopos,&m_nxopos,&imtx);
	D3DXVec3TransformCoord(&nyopos,&m_nyopos,&imtx);

	D3DXVec3TransformCoord(&ncpos,&m_ncpos,&imtx);
	D3DXVec3TransformCoord(&fcpos,&m_fcpos,&imtx);

	//construct the planes

	//near plane
	D3DXPlaneFromPoints(&m_nr,&ncpos,&nxpos,&nypos); 
	D3DXVECTOR3 pnorm(-m_nr.a,-m_nr.b,-m_nr.c);

	//far plane
	D3DXPlaneFromPointNormal(&m_fr,&fcpos,&pnorm); 

	//left plane
	D3DXPlaneFromPoints(&m_lf,&cpos,&cypos,&nxopos);
	
	//right plan
	D3DXPlaneFromPoints(&m_rg,&cypos,&cpos,&nxpos);

	//up plane
	D3DXPlaneFromPoints(&m_up,&cpos,&cxpos,&nypos);
	
	//down plane
	D3DXPlaneFromPoints(&m_dn,&cxpos,&cpos,&nyopos);

	cam.x=imtx._41;
	cam.y=imtx._42;
	cam.z=imtx._43;

	//compute cylinder factors
	ComputeCylynderUp();
}

//-----------------------------------------------------------------------
//
// checks if a vector is in the view frustum 
//
bool ViewFrustum::IsVectorIn(const D3DXVECTOR3 &vec)
{
	// if inside all planes then view frustum :)
	for(int i=0;i<6;i++)
		if (vec.x*plane[i]->a+vec.y*plane[i]->b+vec.z*plane[i]->c+plane[i]->d<0) 
		return false;

	return true;
}

//-----------------------------------------------------------------------
//
// checks if a sphere with center "vec" and radius is in the
// viewfrustum or is outside or intersects it?
//
BYTE ViewFrustum::IsSphereIn(const D3DXVECTOR3 &vec,float radius)
{
	//inital resut = in
	BYTE res=2;
	float f;
	for(int i=0;i<6;i++) {
		//compute distance to plane
		f=vec.x*plane[i]->a+vec.y*plane[i]->b+vec.z*plane[i]->c+plane[i]->d;

		/*		using abs
		float a=(f<0)?-f:f;
		if (a<radius) res=1; //interescts
		else {
			if (f<0) return 0;
		}
		*/

		//avoiding the abs	saves one check... :)
		if (f<0) {
			if (-f < radius) res=1; //intersects
				else return 0; //outside completely
		}
		else {
			if (f < radius) res=1; //intersects
		}
	}
	return res;
}

//-----------------------------------------------------------------------
//
// checks sphere but with squared radius,   ȵ.
//
BYTE ViewFrustum::IsSphereInSQ(const D3DXVECTOR3 &vec,float sqradius)
{
	//inital resut = in
	BYTE res=2;
	float f;
	for(int i=0;i<6;i++) {
		//compute distance to plane
		f=vec.x*plane[i]->a+vec.y*plane[i]->b+vec.z*plane[i]->c+plane[i]->d;

		if (f<0) {
			if (f*f < sqradius) res=1; //intersects
				else return 0; //outside completely
		}
		else {
			if (f*f < sqradius) res=1; //intersects
		}
	}
	return res;
}

//-----------------------------------------------------------------------
//
// compute the radii factors for 6 planes 
//
void ViewFrustum::ComputeCylynderUp()
{	
	for(int i=0;i<6;i++) {
		//up oriented cylinder - the cos is the y component of the normal 
		float fcos=plane[i]->b;
		
		//find the squared sin and the root is the result
		float fsin2=1-(fcos*fcos);
		cyl_plane[i]=sqrtf(fsin2);
	}
}

//-----------------------------------------------------------------------
//
// intersecting the viewfrustum with cylinder up orientated
//
// binf - the BYTE that has the six bits for the six planes
//		  if the bit is 0 for a plane then skip the check
//
BYTE ViewFrustum::IsCylinderUpIn(const D3DXVECTOR3 &center,float radius,float height,BYTE &binf)
{
	BYTE res=2;
	BYTE shf=4;
	//iterate through the six planes
	for(int i=0;i<6;i++,shf<<=1) 
		//if this plane is active
		if (binf&shf) {

			//adjusted radius
			float rr = cyl_plane[i]*radius; 

			//compute the two distances d1 an d2 for the bottom and top 
			//center of the cylinder
			float t=center.x*plane[i]->a+center.z*plane[i]->c+plane[i]->d;;
			float d1 = t+(plane[i]->b*(center.y)); 
			float d2 = t+(plane[i]->b*(center.y+height)); 

			//two flags for the two distances
			bool in1= abs(d1)<rr;
			bool in2= abs(d2)<rr;

			//inetersect 1
			if (in1) {
				res=1;
			} else {
				//inetersect 2
				if (in2) 
						res=1;
				else {
					if (d1<0) {
						if (d2<0) { 
							//update the plane flags
							binf&=(~shf);
							//outside
							return 0;
						} 
						else res=1;
					} else {
						if (d2<0) 
							//intersects
							res=1;
						else 
							//update the plane flags
							binf&=(~shf);
					}
				}
			}
	}
	return res;
}


#ifdef _DEBUG_
extern LPDIRECT3DDEVICE8 g_pD3dDev;
struct FRUSTUMVERTEX
	{
	D3DXVECTOR3 position; // The 3D position for the vertex
	DWORD	color;        // The color for the vertex
	};

// Our custom FVF, which describes our custom vertex structure
#define D3DFVF_FRUSTUMVERTEX (D3DFVF_XYZ|D3DFVF_DIFFUSE)
int inVert=2;
void ViewFrustum::CreateFrustumVertexBuffer()
{
	HRESULT hr=g_pD3dDev->CreateVertexBuffer(inVert*sizeof(FRUSTUMVERTEX),
				D3DUSAGE_DYNAMIC | D3DUSAGE_WRITEONLY,D3DFVF_FRUSTUMVERTEX,
				D3DPOOL_DEFAULT,&m_pVB, NULL);
	FRUSTUMVERTEX *pvb;
	//lock, write, unlock 
	m_pVB->Lock(0,inVert*sizeof(FRUSTUMVERTEX),(void**)&pvb,0);
	memset(pvb,0x00,sizeof(FRUSTUMVERTEX)*inVert);
	pvb[0].color = 0xFF0000FF;
	pvb[1].color = 0xFF0000FF;
	m_pVB->Unlock();

}

void ViewFrustum::UpdateFrustum()
{
	FRUSTUMVERTEX *pvb;
	m_pVB->Lock(0,inVert*sizeof(FRUSTUMVERTEX),(void**)&pvb,0);
	pvb[0].position = *SCamera::GetEye();
	pvb[1].position = *SCamera::GetEye()*50.0f+*SCamera::GetDir();
	pvb[2].position = SCamera::GetEye();
	D3DXMatrixRotationAxis(&mat,&vSide,fCheck);
//	pvb[3].position = SCamera::GetEye()*50.0f+m_current;
	m_pVB->Unlock();
}

void ViewFrustum::Render()
{
	g_pD3dDev->SetFVF( D3DFVF_FRUSTUMVERTEX );
	g_pD3dDev->SetStreamSource(0,m_pVB,0,sizeof(FRUSTUMVERTEX));
	g_pD3dDev->DrawPrimitive(D3DPT_LINELIST, 0, inVert);
}
#endif
